PROGRAM test_sgesv
!
!  Purpose:
!    To test LAPACK subroutine sgesv, which solves a set of N 
!    linear equations in N unknowns.  This test driver also calls 
!    subroutine simul to solve the problem in single precision, 
!    and subroutine dsimul to solve the problem in double precision.  
!    The results of the three solutions together with their errors 
!    are displayed in a summary table.
!
!  Record of revisions:
!      Date       Programmer          Description of change
!      ====       ==========          =====================
!    04/09/96    S. J. Chapman        Modified from program 
!                                        test_dsimul
!
USE booklib, ONLY: simul             ! Use BOOKLIB library
USE lapack_s, ONLY: sgesv            ! Use LAPACK subset library
IMPLICIT NONE

! Declare parameters
INTEGER, PARAMETER :: sgl = SELECTED_REAL_KIND(p=6)   ! Single
INTEGER, PARAMETER :: dbl = SELECTED_REAL_KIND(p=13)  ! Double

! List of local variables
! First, variables for simul solution
REAL(KIND=sgl), ALLOCATABLE, DIMENSION(:,:) :: a
                                 ! Single-precision coefficients
REAL(KIND=sgl), ALLOCATABLE, DIMENSION(:) :: b
                                 ! Single-precision constant values
REAL(KIND=sgl), ALLOCATABLE, DIMENSION(:) :: soln
                                 ! Single-precision solution
REAL(KIND=sgl), ALLOCATABLE, DIMENSION(:) :: serror
                                 ! Array of single-precision errors
INTEGER :: serror_flag           ! Error flag from simul
REAL(KIND=sgl) :: serror_max     ! Max single precision error


! Variables for dsimul solution
REAL(KIND=dbl), ALLOCATABLE, DIMENSION(:,:) :: da
                                 ! Double-precision coefficients
REAL(KIND=dbl), ALLOCATABLE, DIMENSION(:) :: db
                                 ! Double-precision constant values
REAL(KIND=dbl), ALLOCATABLE, DIMENSION(:) :: dsoln
                                 ! Double-precision solution
REAL(KIND=dbl), ALLOCATABLE, DIMENSION(:) :: derror
                                 ! Array of double-precision errors
INTEGER :: derror_flag           ! Error flag from dsimul
REAL(KIND=dbl) :: derror_max     ! Max double precision error


! Variables for single-precision LAPACK solution
INTEGER, ALLOCATABLE, DIMENSION(:) :: ipiv
                                 ! Pivot indices
REAL(KIND=sgl), ALLOCATABLE, DIMENSION(:,:) :: la
                                 ! Single-precision coefficients
REAL(KIND=sgl), ALLOCATABLE, DIMENSION(:) :: lb
                                 ! Single-precision constant values
REAL(KIND=sgl), ALLOCATABLE, DIMENSION(:) :: lerror
                                 ! Array of single-precision errors
INTEGER :: lerror_flag           ! Error flag from sgesv
REAL(KIND=sgl) :: lerror_max     ! Max single precision error
INTEGER :: nrhs = 1              ! Number of right-hand sides


! Other variables
INTEGER :: i, j                  ! Loop index
INTEGER :: istat                 ! I/O status
INTEGER :: n                     ! Size of system of eqns to solve
CHARACTER(len=20) :: filename    ! Input data file name

! Get the name of the disk file containing the equations.
WRITE (*,*) 'Enter the file name containing the eqns: '
READ (*,'(A20)') filename
 
! Open input data file.  Status is OLD because the input data must 
! already exist.
OPEN ( UNIT=1, FILE=filename, STATUS='OLD', ACTION='READ', &
       IOSTAT=istat )
 
! Was the OPEN successful? 
open_ok: IF ( istat == 0 ) THEN
 
   ! The file was opened successfully, so read the number of 
   ! equations in the system.
   READ (1,*) n

   ! Allocate memory for that number of equations
   ALLOCATE ( a(n,n), b(n), soln(n), serror(n), da(n,n), &
              db(n), dsoln(n), derror(n), ipiv(n), la(n,n), &
              lb(n), lerror(n), STAT=istat )

   ! If the memory is available, read in equations and 
   ! process them.
   solve: IF ( istat == 0 ) THEN

      DO i = 1, n
         READ (1,*) (da(i,j), j=1,n), db(i)
      END DO
 
 
      ! Copy the coefficients to single precision for the
      ! single precision solution.
      a  = da
      b  = db
      la = da
      lb = db

      ! Display coefficients.
      WRITE (*,1000)
      1000 FORMAT (/,1X,'Coefficients before calls:')
      DO i = 1, n
         WRITE (*,'(1X,7F11.4)') (a(i,j), j=1,n), b(i)
      END DO
 
      ! Solve equations.
      CALL simul (a,  b,  soln, n, n, serror_flag )
      CALL sgesv (n, nrhs, la, n, ipiv, lb, n, lerror_flag )
      CALL simul (da, db, dsoln, n, n, derror_flag )
 
      ! Display error flags:
      WRITE (*,'(/,A,I6)') ' simul error flag  =', serror_flag
      WRITE (*,'(A,I6)')   ' sgesv error flag  =', lerror_flag
      WRITE (*,'(A,I6)')   ' dsimul error flag =', derror_flag

      ! Check for roundoff by substituting into the original
      ! equations, and calculate the differences.
      serror_max = 0.
      derror_max = 0._dbl
      lerror_max = 0.
      serror = 0.
      lerror_max = 0.
      derror = 0._dbl
      lerror_max = 0. 
      DO i = 1, n         
         serror(i) = sum ( a(i,:)  * soln(:)  ) - b(i)
         derror(i) = sum ( da(i,:) * dsoln(:) ) - db(i)
         lerror(i) = sum ( a(i,:)  * lb(:)  ) - b(i)
      END DO
      serror_max = MAXVAL ( ABS ( serror ) )
      derror_max = MAXVAL ( ABS ( derror ) )
      lerror_max = MAXVAL ( ABS ( lerror ) )

      ! Tell user about it.
      WRITE (*,1010)
      1010 FORMAT (/1X,'  I   SP X(I)    LP X(I)    DP X(I)     ', &
           '   SP ERR       LP ERR       DP ERR  ')
      WRITE (*,1020)
      1020 FORMAT ( 1X,' === ========   ========   ========     ', &
           '  ========     ========     ======== ')
      DO i = 1, n
         WRITE (*,1030) i, soln(i), lb(i), dsoln(i), serror(i), &
                        lerror(i), derror(i)
         1030 FORMAT (1X, I3, 2X, 3G11.5, 3F13.8)
      END DO

      ! Write maximum errors.
      WRITE (*,1040) serror_max, lerror_max, derror_max
      1040 FORMAT (/,1X,'Max simul single-precision error: ',F15.8,&
           /,1X,'Max sgesv single-precision error: ',F15.8, &
           /,1X,'Max dsimul double-precision error:',F15.8)

   END IF solve

   ! Deallocate dynamic memory
   DEALLOCATE ( a, b, soln, serror, da, db, dsoln, derror, &
                ipiv, la, lb, lerror )

ELSE open_ok

   ! Else file open failed.  Tell user.
   WRITE (*,1050) istat
   1050 FORMAT (1X,'File open failed--status = ', I6)

END IF open_ok


END PROGRAM
