PROGRAM spectrum
!
!  Purpose:
!    To calculate and plot the spectrum of an input data set.
!
!  Record of revisions:
!      Date       Programmer          Description of change
!      ====       ==========          =====================
!    03/20/96    S. J. Chapman        Original code
!
USE booklib                           ! Access BOOKLIB
IMPLICIT NONE

! List of parameters:
INTEGER, PARAMETER :: ilu = 9        ! Input data unit
INTEGER, PARAMETER :: maxsiz = 8192  ! Maximum FFT size
INTEGER, PARAMETER :: plu = 6        ! Output plot unit

! List of variables:
COMPLEX,DIMENSION(maxsiz) :: cdata1  ! Complex input data
COMPLEX,DIMENSION(maxsiz) :: cdata2  ! Complex spectrum
REAL,DIMENSION(maxsiz) :: data1      ! Real input data
LOGICAL :: exceed = .FALSE.          ! Array size exceeded 
CHARACTER(len=32) :: filename        ! Input file name
INTEGER :: fft_size                  ! FFT size (power of 2)
REAL :: fs                           ! Sampling freq (Hz)
INTEGER :: i                         ! Index variable
INTEGER :: iexp                      ! Exp of next pwr of 2
INTEGER :: ifreq                     ! Freq to nearest Hz
INTEGER :: nvals                     ! No of vals to process
INTEGER :: status                    ! I/O status
REAL :: temp                         ! Scratch varaible
REAL :: time                         ! Sample time (dummy)
REAL, DIMENSION(0:64) :: xplot       ! Array of freqs to plot
REAL, DIMENSION(0:64) :: yplot       ! Array of spectral values

! Get the name of the file containing the input data.
WRITE (*,*) 'Enter the file name containing the input data: ' 
READ (*,'(A32)') filename

! Get the sampling frequency of the input data.
WRITE (*,*) 'Enter the sampling frequency of the input data: '
READ (*,*) fs
 
! Open input data file.  Status is OLD because the input 
! data must already exist.
OPEN ( ilu, FILE=filename, STATUS='OLD', ACTION='READ', &
       IOSTAT=status )
 
! Was the OPEN successful? 
fileopen: IF ( status == 0 ) THEN       ! Open successful
 
   ! The file was opened successfully, so read the data  
   ! into the program.
   DO
      READ (ilu,*,IOSTAT=status) time, temp  ! Get value
      IF ( status /= 0 ) EXIT              ! Exit on end 
      nvals = nvals + 1                    ! Bump count
      size: IF ( nvals <= maxsiz ) THEN    ! Too many values?
         data1(nvals) = temp               ! No: Save value
      ELSE
         exceed = .TRUE.                   ! Yes: overflow
      END IF size
   END DO
 
   ! Was the array size exceeded?  If so, tell user and quit.
   toobig: IF ( exceed ) THEN
      WRITE (*,1020) nvals, maxsiz
      1020 FORMAT (' Maximum size exceeded: ', I6, ' > ', I6)
   ELSE

      ! Limit not exceeded.  Convert the real input data to 
      ! complex form.
      cdata1(1:nvals) = data1(1:nvals)

      ! Zero pad the data.
      CALL nxtmul ( nvals, 2, iexp, fft_size )
      cdata1(nvals+1:fft_size) = (0.,0.)
      
      ! Call "fft" to calculate complex spectrum.
      CALL fft ( cdata1, cdata2, fft_size, status )
 
      ! Generate (freq, value) pairs to plot at a frequency 
      ! spacing of 1 Hz.  First, place the frequency values 
      ! in each "x" plotting bin.
      xvals: DO i = 0, NINT(fs)
         xplot(i) = REAL(i)
      END DO xvals
 
      ! Calculate the maximum spectral amplitudes for each 
      ! "y" plotting bin.
      yvals: DO i = 1, fft_size
         ! Calculate the freq of the data point to nearest Hz.
         ifreq = NINT ( (fs / REAL(fft_size)) * REAL(i-1) )
         ! Calculate max spectral amplitude for this freq bin.
         yplot(ifreq) = MAX ( yplot(ifreq), CABS(cdata2(i)) )
      END DO yvals
 
      ! Plot spectrum.
      CALL plotxy ( xplot, yplot, NINT(fs), plu )
   END IF toobig

ELSE fileopen
   ! If we get here, the open failed.
   WRITE (*,'(1X,A,A)') 'Open failed on file: ', filename
END IF fileopen

END PROGRAM
