PROGRAM get_signal 
!
!  Purpose:
!    To generate a test signal consisting of a user-specified  
!    sinusoidal signal corrupted by noise.
!
!  Record of revisions:
!      Date       Programmer          Description of change
!      ====       ==========          =====================
!    03/19/96    S. J. Chapman        Original code
!
IMPLICIT NONE

! List of parameters:
INTEGER, PARAMETER :: maxlen = 10000  ! Max length of output

! List of local variables:
REAL :: amp                      ! RMS amplitude of sinusoid
INTEGER :: error                 ! Error flag: 0 = No error
CHARACTER(len=32) :: filename    ! Output file name
REAL :: freq                     ! Frequency of sinusoid in Hz
REAL :: fs                       ! Sampling frequency
INTEGER :: i                     ! Index variable
INTEGER :: nsamp                 ! No. of output samples 
REAL,DIMENSION(maxlen) :: output ! Output signal
REAL :: snr                      ! Desired SNR
INTEGER :: status                ! I/O status flag
REAL :: total_length             ! Total signal length in secs

! Get sinusoid amplitude.
WRITE (*,*) 'Enter the desired amplitude of the sinusoid: '
READ (*,*) amp

! Get sinusoid frequency.
WRITE (*,*) 'Enter the desired freq of the sinusoid, in Hz: '
READ (*,*) freq
 
! Get sampling interval.
WRITE (*,*) 'Enter sampling frequency in Hz: '
READ (*,*) fs
 
! Get total signal length, in seconds.
WRITE (*,*) 'Enter total signal length in seconds: '
READ (*,*) total_length

! Get SNR in dB.
WRITE (*,*) 'Enter desired SNR in dB: '
READ (*,*) snr

! Get output file name.
WRITE (*,*) 'Enter output file name: '
READ (*,'(A)') filename
 
! Open output file on unit 7.
OPEN ( 7, FILE=filename, STATUS='REPLACE', ACTION='WRITE', &
       IOSTAT=status )

! Was the file open successful?
openok: IF ( status == 0 ) THEN
 
   ! Yes.  Calculate the resulting test data set.
   CALL test_signal ( amp, freq, fs, total_length, snr, &
                      maxlen, output, nsamp, error )
 
   ! Check for errors in subroutine test_signal.
   result: IF ( error /= 0 ) THEN 
      WRITE (*,*) 'Error in subroutine test_signal: no data.'
   ELSE
      ! Data is OK.  Write out results.
      DO i = 1, nsamp
         WRITE (7,'(1X,F14.6,3X,F14.6)') REAL(i-1)/fs, output(i)
      END DO
   END IF result

ELSE openok
   ! A file OPEN error occurred.
   WRITE (*,'(3A,I6)') 'Open error in file ', filename, &
                       ' STATUS = ', status
END IF openok

! Close output file
CLOSE (UNIT=7, STATUS='KEEP')

END PROGRAM

SUBROUTINE test_signal ( amp, freq, fs, total_length, snr, &
                         maxlen, output, nsamp, error )
!
!  Purpose:
!    To generate a test signal consisting of a user-specified 
!    sinusoidal signal corrupted by noise.
!
!  Record of revisions:
!      Date       Programmer          Description of change
!      ====       ==========          =====================
!    03/19/96    S. J. Chapman        Original code
!
USE booklib                      ! Use BOOKLIB library
IMPLICIT NONE

! List of parameters: 
REAL, PARAMETER :: pi = 3.141593 ! pi

! List of calling arguments:
REAL, INTENT(IN) :: amp          ! RMS amp of sinusoid 
REAL, INTENT(IN) :: freq         ! Frequency of sinusoid, Hz
REAL, INTENT(IN) :: fs           ! Sampling frequency, Hz
REAL, INTENT(IN) :: total_length ! Total signal length, sec
REAL, INTENT(IN) :: snr          ! Desired SNR
INTEGER, INTENT(IN) :: maxlen    ! Length of array "output"
REAL,DIMENSION(maxlen), INTENT(OUT) :: output ! Output signal
INTEGER, INTENT(OUT) :: nsamp    ! No. of output samples
INTEGER, INTENT(OUT) :: error    ! Error flag: 0 = No error

! List of local variables: 
REAL :: amp_noise                ! Noise amplitude
REAL :: factor                   ! 2 * pi * freq / fs
INTEGER :: i                     ! Index variable
REAL :: peak_amp                 ! Peak Signal Amplitude
 
! Calculate nsamp and noise amplitude
nsamp  = NINT ( total_length * fs ) + 1
amp_noise = amp / ( 10.**(snr/20.) )
 
! Is there enough room to place the data in the output array?
IF ( nsamp > maxlen ) THEN 
 
   ! No--Set error condition and get out.
   error = 1
   nsamp = 0
 
ELSE
 
   ! Yes--Calculate the output data array.
   factor = 2. * pi * freq / fs 
   peak_amp  = SQRT(2.) * amp
   DO i = 1, nsamp
      output(i) = peak_amp * SIN ( factor * REAL(i-1) ) &
                + amp_noise * random_n()  
   END DO
END IF

END SUBROUTINE test_signal
